#ifndef __CMathTools__
#define __CMathTools__

#include "../Basics/CCountedObject.hpp"
#include <algorithm>
using namespace std;

#ifdef WIN32
	#define IMAN 0
#else
	#ifdef __i386
		#define IMAN 0
	#else
		#define IMAN 1
	#endif
#endif

#ifdef WIN32
	#define exponent_position 1
#else
	#define exponent_position 0
#endif


static float shift23=(1<<23);
static float PowBodge=0.33971f;

//	===========================================================================

namespace Exponent
{
	namespace MathTools
	{
		/**
		 * @class CMathTools CMathTools.hpp
		 * @brief A few handy maths functions. Sppedy version of std c++ functions are prefixed fastXXX
		 *
		 * @date 14/11/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @deprecated Soon to be replaced with a template version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CMathTools.hpp,v 1.9 2007/02/08 21:06:44 paul Exp $
		 */
		class CMathTools
		{
		public:

//	===========================================================================

			const static float CMATH_HALF_PI_FLOAT;				/**< pi * 0.5 */
			const static float CMATH_PI_FLOAT;					/**< pi */
			const static float CMATH_2PI_FLOAT;					/**< pi * 2 */
			const static float CMATH_4PI_FLOAT;					/**< pi * 4 */

//	===========================================================================

			const static double CMATH_HALF_PI_DOUBLE;			/**< pi * 0.5 */
			const static double CMATH_PI_DOUBLE;				/**< pi */
			const static double CMATH_2PI_DOUBLE;				/**< pi * 2 */
			const static double CMATH_4PI_DOUBLE;				/**< pi * 4 */

//	===========================================================================

			const static double CMATH_LN2_DOUBLE;				/**< ln(2) */
			const static double CMATH_LN2_INV_DOUBLE;			/**< 1/ln(2) */

//	===========================================================================

			const static float CMATH_LN2_FLOAT;					/**< ln(2) */
			const static float CMATH_LN2_INV_FLOAT;				/**< 1/ln(2) */

//	===========================================================================

			const static double CMATH_ONE_THIRD_DOUBLE;			/**< 1/3 */
			const static double CMATH_TWO_THIRDS_DOUBLE;		/**< 2/3 */

//	===========================================================================

			const static float CMATH_ONE_THIRD_FLOAT;			/**< 1/3 */
			const static float CMATH_TWO_THIRDS_FLOAT;			/**< 2/3 */

//	===========================================================================

			/**
			 * Get the minimum of two values
			 * @param a The A Value
			 * @param b The B Value
			 * @retval double min(a,b);
			 */
			static FORCEINLINE double minimum(const double a, const double b)
			{
				return min(a, b);
			}

			/**
			 * Get the minimum of two values
			 * @param a The A Value
			 * @param b The B Value
			 * @retval float min(a,b);
			 */
			static FORCEINLINE float minimum(const float a, const float b)
			{
				return min(a, b);
			}

			/**
			 * Get the minimum of two values
			 * @param a The A Value
			 * @param b The B Value
			 * @retval long min(a,b);
			 */
			static FORCEINLINE long minimum(const long a, const long b)
			{
				return min(a, b);
			}

			/**
			 * Get the minimum of two values no branch - By quintosardo
			 * @param a The A Value
			 * @param b The B Value
			 * @retval double min(a,b);
			 */
			static FORCEINLINE double fastMinimum(const double a, const double b)
			{
				return 0.5 * (a + b - fabs(a - b));
			}

			/**
			 * Get the minimum of two values no branch - By quintosardo
			 * @param a The A Value
			 * @param b The B Value
			 * @retval float min(a,b);
			 */
			static FORCEINLINE float fastMinimum(const float a, const float b)
			{
				return 0.5f * (a + b - fabsf(a - b));
			}

//	===========================================================================

			/**
			 * Get the maximum of two values
			 * @param a The A Value
			 * @param b The B Value
			 * @retval double max(a,b);
			 */
			static FORCEINLINE double maximum(const double a, const double b)
			{
				return max(a, b);
			}

			/**
			 * Get the maximum of two values
			 * @param a The A Value
			 * @param b The B Value
			 * @retval float max(a,b);
			 */
			static FORCEINLINE float maximum(const float a, const float b)
			{
				return max(a, b);
			}

			/**
			 * Get the maximum of two values
			 * @param a The A Value
			 * @param b The B Value
			 * @retval long max(a,b);
			 */
			static FORCEINLINE long maximum(const long a, const long b)
			{
				return max(a, b);
			}

			/**
			 * Get the maximum no branching - By quintosardo
			 * @param a The A Value
			 * @param b The B Value
			 * @retval double max(a,b);
			 */
			static FORCEINLINE double fastMaximum(const double a, const double b)
			{
				return 0.5 * (a + b + fabs(a - b));
			}

			/**
			 * Get the maximum no branching - By quintosardo
			 * @param a The A Value
			 * @param b The B Value
			 * @retval float max(a,b);
			 */
			static FORCEINLINE float fastMaximum(const float a, const float b)
			{
				return 0.5f * (a + b + fabsf(a - b));
			}

//	===========================================================================

			/**
			 * Get the floating point component of a value
			 * @param x The value to get the floating point component from
			 * @retval double The floating point component
			 */
			static FORCEINLINE double fastModulus(const double x)
			{
				return x - floor(x);
			}

			/**
			 * Get the floating point component of a value
			 * @param x The value to get the floating point component from
			 * @retval float The floating point component
			 */
			static FORCEINLINE float fastModulus(const float x)
			{
				return x - floorf(x);
			}

//	===========================================================================

			/**
			 * Fast atan - By antiprosynthesis
			 * @note Documented online at <A HREF="http://www.musicdsp.org/showone.php?id=104">http://www.musicdsp.org/showone.php?id=104</A>
			 * @param x The phase in radians
			 * @retval double atan(x)
			 */
			static FORCEINLINE double fastAtan(const double x)
			{
				return (x / (1.0 + 0.28 * (x * x)));
			}

			/**
			 * Fast atan - By antiprosynthesis
			 * @note Documented online at <A HREF="http://www.musicdsp.org/showone.php?id=104">http://www.musicdsp.org/showone.php?id=104</A>
			 * @param x The phase in radians
			 * @retval float atan(x)
			 */
			static FORCEINLINE float fastAtan(const float x)
			{
				return (x / (1.f + 0.28f * (x * x)));
			}

//	===========================================================================

			/**
			 * Clamp to 0 - 1 - By Duncan S Parsons
			 * @param min The minimum of the value
			 * @param max The maximum of the value
			 * @param x The value
			 * @retval double x in range min - max
			 */
			static FORCEINLINE double clamp(const double min, const double max, const double x)
			{
				return 0.5 * (fabs(x - min) + (min + max) - fabs(x - max));
			}

			/**
			 * Clamp to 0 - 1 - By Duncan S Parsons
			 * @param x The value
			 * @retval float x in range min - max
			 */
			static FORCEINLINE double clamp(const double x)
			{
				return 0.5 * (fabs(x) + 1.0 - fabs(x - 1.0));
			}

			/**
			 * Clamp to a range - By Duncan S Parsons
			 * @param min The minimum of the value
			 * @param max The maximum of the value
			 * @param x The value
			 * @retval float x in range min - max
			 */
			static FORCEINLINE float clamp(const float min, const float max, const float x)
			{
				return 0.5f * (fabsf(x - min) + (min + max) - fabsf(x - max));
			}

//	===========================================================================

			/**
			 * Cube a number
			 * @param x The number to square
			 * @retval double x*x*x
			 */
			static FORCEINLINE double cube(const double x)
			{
				return x * x * x;
			}

			/**
			 * Cube a number
			 * @param x The number to square
			 * @retval float x*x*x
			 */
			static FORCEINLINE float cube(const float x)
			{
				return x * x * x;
			}

//	===========================================================================

			/**
			 * Square a number
			 * @param x The number to square
			 * @retval double x*x
			 */
			static FORCEINLINE double square(const double x)
			{
				return x * x;
			}

			/**
			 * Square a number
			 * @param x The number to square
			 * @retval float x*x
			 */
			static FORCEINLINE float square(const float x)
			{
				return x * x;
			}

//	===========================================================================

			/**
			 * Double to Long conversion
			 * @note Documented online at <A HREF="http://www.stereopsis.com/FPU.html#convert">http://www.stereopsis.com/FPU.html#convert</A>
			 * @param x The value to convert to a long
			 * @retval long (long)x;
			 */
			static FORCEINLINE long fastDoubleToLong(double x)
			{
			#ifdef WIN32
				x = x + 103079215104;
			#else
				#ifndef __i386
					x = x + 103079215104.0;
				#else
					return (long)x;
				#endif
			#endif
				return ((long*)&x)[IMAN] >> 16; 
			}

			/**
			 * Fast floor
			 * @note Original by Laurent De Soras with additions by Daniel Schaack
			 * @note Documented online at <A HREF="http://www.musicdsp.org/showone.php?id=170">http://www.musicdsp.org/showone.php?id=170</A>
			 * @param x The value to floor
			 * @retval long The floored value
			 */
			static FORCEINLINE long fastDoubleToLongFloor(const double x)
			{
			#ifdef WIN32
				long returnValue;
				__asm
				{
					fld x
					fistp returnValue
				}
				return returnValue;
			#else
				return (long)floor(x);
			#endif
			}

//	===========================================================================
			
			/**
			 * Compute bessel function iZero using a series approximation
			 * @param y value to compute iZero for
			 * @retval double iZero
			 * @note Algorithm from C++ algorithms for DSP by Embree & Danielli
			 */
			static FORCEINLINE double iZero(const double y)
			{
				double s  = 1.0;
				double ds = 1.0;
				double d  = 0.0;
				do
				{
					d  = d + 2.0;
					ds = ds * (y * y) / (d * d);
					s  = s + ds;
				}while(ds > 1e-7 * s);
				return s;
			}

			/**
			 * Compute the zeroth order bessel
			 * @param x The input value
			 * @retval double The output zeroth order bessel
			 * @note based on the observations on <A HREF="http://en.wikipedia.org/wiki/Bessel_Functions">wikipedia</A>
			 */
			static FORCEINLINE double zeroethOrderBessel(const double x)
			{
				double bessel = 1.0;
				double term   = 0.0;
				long i		  = 1;

				do
				{
					// Compute the term
					term = pow(0.5 * x, (double)i) / factorial((double)i);

					// Add to bessel
					bessel += (term * term);

					// Increment to next
					i++;
				}while (term > 0.000001 * bessel);

				// Return the bessel value
				return bessel;
			}

			/**
			 * Compute the factorial of a number
			 * @param x The value
			 * @retval double The factorial
			 */
			static FORCEINLINE double factorial(const double x)
			{
				// Store some values
				double outputValue = 1.0;	
				double value       = x;

				// Compute factorial
				while (value > 1) 
				{
    				outputValue *= value--;
				}

				// Done :)
				return outputValue;
			}

//	===========================================================================

			/** @cond */

//	===========================================================================

			/**
			 * Fast sign
			 * @note Original by Tobybear
			 * @note Documented online at <A HREF="http://www.musicdsp.org/showone.php?id=132">http://www.musicdsp.org/showone.php?id=132</A>
			 * @param x The value to get the sign of
			 * @retval int -1 if x<0 0 if x=0 +1 if x>0
			 */
			static FORCEINLINE int fastSign(const float x)
			{
			#ifdef WIN32
				return 1 + (((*(int *) &x) >> 31) << 1);
			#else
				if (x > 0)
				{
					return 1;
				}
				else if (x < 0)
				{
					return -1;
				}
				return 0;
			#endif
			}

			/**
			 * Fast sign
			 * @note Original by Tobybear
			 * @note Documented online at <A HREF="http://www.musicdsp.org/showone.php?id=132">http://www.musicdsp.org/showone.php?id=132</A>
			 * @param x The value to get the sign of
			 * @retval int -1 if x<0 0 if x=0 +1 if x>0
			 */
			static FORCEINLINE int fastSign(const double x)
			{
			#ifdef WIN32
				return fastSign((float)x);
			#else
				if (x > 0)
				{
					return 1;
				}
				else if (x < 0)
				{
					return -1;
				}
				return 0;
			#endif
			}

//	===========================================================================

			/**
			 * Fast log
			 * @param x The value to compute the log of
			 * @retval double log(x)
			 */
			static FORCEINLINE double fastLog(const double x)
			{
			#ifdef WIN32
				return (double)fastLog((float)x);
			#else
				return logf(x);
			#endif
			}

			/**
			 * Fast log
			 * @param x The value to compute the log of
			 * @retval float log(x)
			 */
			static FORCEINLINE float fastLog(const float x)
			{
			#ifdef WIN32
				return (fastLog2(x) * 0.69314718f);
			#else
				return log(x);
			#endif
			}

//	===========================================================================

			/**
			 * Fast abs - By TobyBear
			 * @param x The value to compute the absolute of
			 * @retval double abs(x)
			 */
			static FORCEINLINE double fastAbsolute(const double x)
			{
			#ifdef WIN32
				return (double)fastAbsolute((float)x);
			#else
				return fabs(x);
			#endif
			}

			/**
			 * Fast abs - By TobyBear
			 * @param x The value to compute the absolute of
			 * @retval float abs(x)
			 */
			static FORCEINLINE float fastAbsolute(const float x)
			{
			#ifdef WIN32
				int i = ((*(int*)&x)&0x7fffffff);
				return (*(float*)&i);
			#else
				return fabsf(x);
			#endif
			}

//	===========================================================================


			/**
			 * Fast square root
			 * @note Documented online at <A HREF="http://www.azillionmonkeys.com/qed/sqroot.html">http://www.azillionmonkeys.com/qed/sqroot.html</A>
			 * @param x The value to square root
			 * @retval double sqrt(x)
			 */
			static FORCEINLINE double fastSquareRoot(const double x)
			{
			#ifdef WIN32
				double y, z, temp;
				unsigned long *tempPtr = ((unsigned long *)&temp)+1;

				temp = x;
				*tempPtr = (0xbfcd4600 - *tempPtr) >> 1;
				y = temp;
				z = x * 0.5;
				y *= 1.5 - y * y * z;
				y *= 1.5 - y * y * z;
				y *= 1.5 - y * y * z;
				y *= 1.5 - y * y * z;
				return y * x;
			#else
				return sqrt(x);
			#endif
			}

			/**
			 * Fast square root
			 * @note Documented online at <A HREF="http://www.azillionmonkeys.com/qed/sqroot.html">http://www.azillionmonkeys.com/qed/sqroot.html</A>
			 * @param x The value to square root
			 * @retval float sqrt(x)
			 */
			static FORCEINLINE float fastSquareRoot(const float x)
			{
			#ifdef WIN32
				return (float)fastSquareRoot((double)x);
			#else
				return sqrtf(x);
			#endif
			}


			/**
			 * Fast square root - From KVR - untested...
			 * @note Im actively hunting for the thread that i got this from. I cant find it using the search so if anyone comes across the thread please send you answers on the back of a sealed envelope ;)
			 * @param x The value to square root
			 * @retval float sqrt(x)
			 */
			static FORCEINLINE float fastSquareRootVersion2(const float x)
			{
			#ifdef WIN32
				float   result;
				_asm
				{
					mov eax, x
					sub eax, 0x3f800000
					sar eax, 1
					add eax, 0x3f800000
					mov result, eax
				}
				return result;
			#else
				return sqrtf(x);
			#endif
			}

//	===========================================================================

			/**
			 * Fast exponential computation\n
			 * based upon the premise that exp(x*ln2)=pow(2, x)\n
			 * and given that\n
			 * exp(x)=(2^(1/ln2))^x
			 */
			static FORCEINLINE double fastExponential(const double x)
			{
				return myPow2(x * CMATH_LN2_INV_DOUBLE);
			}


			static FORCEINLINE float myPow2(float i)
			{
				//float PowBodge=0.33971f;
				float x;
				float y=i-floorf(i);
				y=(y-y*y)*PowBodge;

				x=i+127-y;
				x*= shift23; //pow(2,23);
				*(int*)&x=(int)x;
				return x;
			}

			static FORCEINLINE double myPow2(double x)
			{
				// store address of float as long pointer
				long *p = (long*)(&x) + exponent_position;
				// truncated x for integer power of 2
				const long tx = fastDoubleToLongFloor(x);
				// float remainder of power of 2
				x -= static_cast<double>(tx);
				// sqr apporoximation of 2^x in [0, 1]
				//x  = 1.0 + x*(2.0/3.0 + x*1.0/3.0);
				x = 1.0 + x * (0.6 + x * 0.3);
				// add integer power of 2 to exponent
				*p += (tx<<20);
				return x;
			}

			/**
			 * Fast exp2 - By Laurent de Soras
			 * @param x The value to calculate the exp2 of
			 * @retval double exp2(x);
			 */
			static FORCEINLINE double fastExp2(const double x)
			{
			#ifdef WIN32
				int e;
				double ret;

				if (x >= 0)
				{
					e = int (x);
					ret = x - (e - 1);
					((*(1 + (int *) &ret)) &= ~(2047 << 20)) += (e + 1023) << 20;
				}
				else
				{
					e = int (x + 1023);
					ret = x - (e - 1024);
					((*(1 + (int *) &ret)) &= ~(2047 << 20)) += e << 20;
				}
				return (ret);
			#else
				return exp2(x);
			#endif
			}

			/**
			 * Fast exp2 - By Laurent de Soras
			 * @param x The value to calculate the exp2 of
			 * @retval float exp2(x);
			 */
			static FORCEINLINE float fastExp2(const float x)
			{
			#ifdef WIN32
				return (float)fastExp2((double)x);
			#else
				return exp2f(x);
			#endif
			}

			/**
			 * Fast Expontential - based upon the paper\n
			 * "On a Fast, Compact Approximation of the Exponential Function"\n
			 * by Gavin C. Cawley vailable online at :\n
			 * http://www.cacr.caltech.edu/~patrickh/documents/pdf/2009.pdf\n
			 * @param x The value to get the exp of
			 * @retval double exp(x)
			 */
			static FORCEINLINE double fastExp(const double x)
			{
				/*

				y = 2


				exp(x * ln(2)) == pow(2, x)

				I want a fast exponential function....

				pow(2, x) == exp(x * ln(2)
				pow(2, (x / ln2)) == exp(x)

				2^x = e^xln(2)




				
				// exp(x * ln(y)) == pow(y, x)

				*/
				// Nameless union
				union
				{
					double d;
				#ifdef WIN32
					struct { int j, i; } n;
				#else
					struct { int i, j; } n;
				#endif
				}eco;

				eco.n.i = (int)(1512775.3951951856938358403823062 * x) + (1072632447);	//(1072693248 - 60801);
				eco.n.j = 0;

				return eco.d;
			}

//	===========================================================================

			/**
			 * Fast x^4	- By Stefan Stenzel
			 * @param x The value to compute
			 * @retval double x^4
			 */
			static FORCEINLINE double fastPower4(double x)
			{
			#ifdef WIN32
				return (double)fastPower4((float)x);
			#else
				return x * x * x * x;
			#endif
			}

			/**
			 * Fast x^4	- By Stefan Stenzel
			 * @param x The value to compute
			 * @retval float x^4
			 */
			static FORCEINLINE float fastPower4(float x)
			{
			#ifdef WIN32
				long *longPointer = (long *)(&x);
				long myLong   	  = *longPointer;
				myLong -= 0x3F800000l;	// Unbias
				myLong <<= 2;			// ^4
				myLong += 0x3F800000l;	// Rebias
				*longPointer = myLong;
				return x;
			#else
				return x * x * x * x;
			#endif
			}

			/**
			 * Fast power
			 * Suggested by Mike Deskevich on http://www.codeguru.com/forum/archive/index.php/t-101010.html
			 * @param b The base
			 * @param e The exponent
			 * @retval double b^e
			 */
			static FORCEINLINE double fastPower(const double b, const double e)
			{
				return exp(e * log(b));
			}

//	===========================================================================

			/**
			 * Fast sin
			 * @param x phase in radians 0 - PI/2
			 * @retval double sin(x)
			 */
			static FORCEINLINE double fastSin(const double x)
			{
				const double xSquared = x * x;
				double returnValue	  = 0.00761;
				returnValue			 *= xSquared;
				returnValue			 -= 0.16605;
				returnValue			 *= xSquared;
				returnValue			 += 1.0;
				returnValue			 *= x;
				return returnValue;
			}

//	===========================================================================

			/**
			 * Fast Tanh
			 * @param x The phase in randians -2pi to 2pi or -1 to +1
			 * @retval double tanh[x]
			 */
			static FORCEINLINE double tanhApproximation(const double x)
			{
				double a = fabs(x);
				a*=(6+a*(3+a));
				return ((x<0)?-1:1)*a/(a+12);
				//a = 6.0 + a * (6.0 + a * (3.0 + a));
				//return ((x < 0) ? -1 : 1) * (a - 6.0) / (a + 6.0);
				//return 1.0 - 2.0 / (fastExponential(2.0 * x) + 1.0);
			}

			/**
			 * Fast tanh - By Fuzzpilz
			 * @param x The phase in radians
			 * @retval double The tanh of x
			 */
			static FORCEINLINE double fastTanh(const double x)
			{
				//return 1.0 - 2.0 / (exp(2.0 * x) + 1);
				return 1.0 - 2.0 / (fastExp(2.0 * x) + 1.0);
			}

			/**
			 * Fast tanh - By Fuzzpilz
			 * @param x The phase in radians
			 * @retval float The tanh of x
			 */
			static FORCEINLINE float fastTanh(const float x)
			{
				return 1.f - 2.f / (expf(2.f * x) + 1.f);
			}

#ifdef WIN32

			/**
			 * Fast tanh - By Fuzzpilz
			 * @param x The phase in radians
			 * @retval double The tanh of x
			 */
			static FORCEINLINE double fastTanh2(const double x)
			{
				double absolute = fabs(x);
				absolute *= (6 + absolute * (3 + absolute));
				return fastSign(x) * absolute / (absolute + 12);
			}

			/**
			 * Fast tanh - By Fuzzpilz
			 * @param x The phase in radians
			 * @retval double Tanh[x/2]
			 */
			static FORCEINLINE float fastTanh2(const float x)
			{
				float absolute = fabsf(x);
				absolute *= (6 + absolute * (3 + absolute));
				return fastSign(x) * absolute / (absolute + 12);
			}
#endif

//	===========================================================================

		protected:

//	===========================================================================
#ifdef WIN32
			/**
			 * Fast log2 (value MUST BE STRICTLY > 0.f) - By Laurent de Soras
			 * @param x The value to log2
			 * @retval float log2(x)
			 */
			static FORCEINLINE float fastLog2(float x)
			{
			   	int * const  exp_ptr = reinterpret_cast <int *> (&x);
			   	int val 			 = *exp_ptr;
			   	const int log_2 	 = ((val >> 23) & 255) - 128;
			   	val &= ~(255 << 23);
			   	val += 127 << 23;
			   	*exp_ptr = val;

   				return (x + log_2);
			}
#endif

			/** @endcond */

		};
	}
}
#endif	// End of CMathTools.hpp